import { createClient } from '@supabase/supabase-js';
import { 
  Lab, Reagen, StokReagenLot, LogPemakaianReagen, Alat, 
  PeminjamanAlat, KalibrasiAlat, AlatStatus, PeminjamanStatus 
} from '../types';

// Supabase Configuration
const SUPABASE_URL = 'https://ssnwsmjuzvancyayxmws.supabase.co';

// WARNING: Menggunakan Service Role Key (Secret) di client-side sangat berbahaya untuk production.
// Ini hanya digunakan untuk debugging agar bisa bypass RLS (Row Level Security).
const SUPABASE_KEY = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InNzbndzbWp1enZhbmN5YXl4bXdzIiwicm9sZSI6InNlcnZpY2Vfcm9sZSIsImlhdCI6MTc2OTk5MjIxNiwiZXhwIjoyMDg1NTY4MjE2fQ.fu4z-Eg2kF0-8MmyvNnkjmmJEYaeGU27GFC8u9-U12w';

export const supabase = createClient(SUPABASE_URL, SUPABASE_KEY, {
  auth: {
    persistSession: true,
    autoRefreshToken: true,
  }
});

class SupabaseService {
  // --- Auth ---
  async signIn(email: string, password: string) {
    return await supabase.auth.signInWithPassword({ email, password });
  }

  async signOut() {
    return await supabase.auth.signOut();
  }

  // --- Labs ---
  async getLabs(): Promise<Lab[]> {
    const { data, error } = await supabase.from('laboratorium').select('*').order('nama_lab');
    if (error) throw error;
    return data || [];
  }

  async addLab(nama_lab: string): Promise<void> {
    const { error } = await supabase.from('laboratorium').insert({ nama_lab });
    if (error) throw error;
  }

  // MANUAL CASCADE: Delete children first
  async deleteLab(id: string): Promise<void> {
    // 1. Delete Reagents (and their lots/logs via deleteReagent logic)
    const { data: reagents } = await supabase.from('reagen').select('id').eq('lab_id', id);
    if (reagents) {
      for (const r of reagents) {
        await this.deleteReagent(r.id);
      }
    }

    // 2. Handle Tools: Update to null or delete?
    // User expectation: Cascade delete.
    const { data: tools } = await supabase.from('alat').select('id').eq('lab_id', id);
    if (tools) {
      for (const t of tools) {
        await this.deleteTool(t.id);
      }
    }

    const { error } = await supabase.from('laboratorium').delete().eq('id', id);
    if (error) throw error;
  }

  // --- Reagents ---
  async getReagents(labId?: string): Promise<Reagen[]> {
    let query = supabase.from('reagen').select('*').order('nama_reagen');
    if (labId) query = query.eq('lab_id', labId);
    
    const { data, error } = await query;
    if (error) throw error;
    return data || [];
  }

  async addReagent(reagen: Omit<Reagen, 'id'>): Promise<void> {
    const { error } = await supabase.from('reagen').insert(reagen);
    if (error) throw error;
  }

  // MANUAL CASCADE
  async deleteReagent(id: string): Promise<void> {
    // 1. Get Lots
    const { data: lots } = await supabase.from('stok_reagen_lot').select('id').eq('reagen_id', id);
    if (lots) {
      for (const lot of lots) {
        // 2. Call deleteLot which handles logs
        await this.deleteLot(lot.id);
      }
    }
    
    const { error } = await supabase.from('reagen').delete().eq('id', id);
    if (error) throw error;
  }

  // --- Lots ---
  async getLots(reagenId?: string): Promise<StokReagenLot[]> {
    let query = supabase.from('stok_reagen_lot').select('*').order('tgl_expired');
    if (reagenId) query = query.eq('reagen_id', reagenId);

    const { data, error } = await query;
    if (error) throw error;
    return data || [];
  }

  async getAllLots(): Promise<StokReagenLot[]> {
    const { data, error } = await supabase.from('stok_reagen_lot').select('*');
    if (error) throw error;
    return data || [];
  }

  async addLot(lot: Omit<StokReagenLot, 'id'>): Promise<void> {
    const { error } = await supabase.from('stok_reagen_lot').insert(lot);
    if (error) throw error;
  }

  // MANUAL CASCADE
  async deleteLot(id: string): Promise<void> {
    // 1. Delete Logs first
    await supabase.from('log_pemakaian_reagen').delete().eq('lot_id', id);

    // 2. Delete Lot
    const { error } = await supabase.from('stok_reagen_lot').delete().eq('id', id);
    if (error) throw error;
  }

  // --- Logs & Stock Logic (Transaction Simulation) ---
  async getLogsByLot(lotId: string): Promise<LogPemakaianReagen[]> {
    const { data, error } = await supabase
      .from('log_pemakaian_reagen')
      .select('*, stok_reagen_lot(no_lot)')
      .eq('lot_id', lotId)
      .order('tgl_input', { ascending: false });

    if (error) throw error;
    return data || [];
  }

  async addUsageLog(log: Omit<LogPemakaianReagen, 'id' | 'tgl_input'>): Promise<void> {
    // 1. Get current stock
    const { data: lot, error: lotError } = await supabase
      .from('stok_reagen_lot')
      .select('stok_sekarang')
      .eq('id', log.lot_id)
      .single();
    
    if (lotError || !lot) throw new Error("Lot not found");
    if (lot.stok_sekarang < log.jumlah_digunakan) throw new Error("Stok tidak cukup");

    // 2. Insert Log
    const { error: logError } = await supabase.from('log_pemakaian_reagen').insert({
      lot_id: log.lot_id,
      jumlah_digunakan: log.jumlah_digunakan,
      keterangan: log.keterangan
    });
    if (logError) throw logError;

    // 3. Decrement Stock
    const { error: updateError } = await supabase
      .from('stok_reagen_lot')
      .update({ stok_sekarang: lot.stok_sekarang - log.jumlah_digunakan })
      .eq('id', log.lot_id);

    if (updateError) {
        console.error("Stock update failed, manual intervention needed");
    }
  }

  async deleteUsageLog(logId: string): Promise<void> {
    // 1. Get Log Details
    const { data: log, error: logError } = await supabase
      .from('log_pemakaian_reagen')
      .select('*')
      .eq('id', logId)
      .single();
    
    if (logError || !log) throw new Error("Log not found");

    // 2. Get Current Stock
    const { data: lot } = await supabase
      .from('stok_reagen_lot')
      .select('stok_sekarang')
      .eq('id', log.lot_id)
      .single();

    // 3. Increment Stock (Revert) ONLY if Lot exists
    if (lot) {
      const { error: updateError } = await supabase
        .from('stok_reagen_lot')
        .update({ stok_sekarang: lot.stok_sekarang + log.jumlah_digunakan })
        .eq('id', log.lot_id);

      if (updateError) throw updateError;
    }

    // 4. Delete Log
    const { error: deleteError } = await supabase
      .from('log_pemakaian_reagen')
      .delete()
      .eq('id', logId);

    if (deleteError) throw deleteError;
  }

  // --- Equipment ---
  async getAllTools(): Promise<Alat[]> {
    const { data, error } = await supabase.from('alat').select('*').order('nama_alat');
    if (error) throw error;
    return data || [];
  }

  async addTool(tool: Omit<Alat, 'id'>): Promise<void> {
    const { error } = await supabase.from('alat').insert(tool);
    if (error) throw error;
  }

  // MANUAL CASCADE
  async deleteTool(id: string): Promise<void> {
    // 1. Delete Loans
    await supabase.from('peminjaman_alat').delete().eq('alat_id', id);
    // 2. Delete Calibrations
    await supabase.from('kalibrasi_alat').delete().eq('alat_id', id);
    
    // 3. Delete Tool
    const { error } = await supabase.from('alat').delete().eq('id', id);
    if (error) throw error;
  }

  async borrowTool(alatId: string, peminjam: string): Promise<void> {
    const { error: toolError } = await supabase
      .from('alat')
      .update({ status: AlatStatus.DIPINJAM })
      .eq('id', alatId);
    
    if (toolError) throw toolError;

    const { error: loanError } = await supabase.from('peminjaman_alat').insert({
      alat_id: alatId,
      peminjam: peminjam,
      status_kembali: PeminjamanStatus.DIPINJAM
    });

    if (loanError) throw loanError;
  }

  // --- Calibration ---
  async getAllCalibrations(): Promise<KalibrasiAlat[]> {
    const { data, error } = await supabase.from('kalibrasi_alat').select('*').order('tgl_expired_kalibrasi');
    if (error) throw error;
    return data || [];
  }

  async addCalibration(cal: Omit<KalibrasiAlat, 'id'>): Promise<void> {
    const { error } = await supabase.from('kalibrasi_alat').insert(cal);
    if (error) throw error;
  }
}

export const db = new SupabaseService();