import React, { useState, useEffect } from 'react';
import { db } from '../services/mockSupabase';
import { Alat, KalibrasiAlat, AlatStatus, Lab } from '../types';
import { Wrench, Search, FileCheck, UploadCloud, Plus, Trash2 } from 'lucide-react';

const EquipmentManagement: React.FC = () => {
  const [activeTab, setActiveTab] = useState<'list' | 'calibration'>('list');
  const [tools, setTools] = useState<Alat[]>([]);
  const [labs, setLabs] = useState<Lab[]>([]);
  
  // Forms
  const [showAddTool, setShowAddTool] = useState(false);
  const [newTool, setNewTool] = useState({ lab_id: '', nama: '', merk: '', tipe: '' });

  // Calibration State
  const [calibrations, setCalibrations] = useState<KalibrasiAlat[]>([]);

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    const [t, l] = await Promise.all([db.getAllTools(), db.getLabs()]);
    setTools(t);
    setLabs(l);
  };

  useEffect(() => {
    if (activeTab === 'calibration') {
      db.getAllCalibrations().then(setCalibrations);
    }
  }, [activeTab]);

  const handleAddTool = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!newTool.lab_id) {
      alert("Pilih Laboratorium");
      return;
    }
    await db.addTool({
      lab_id: newTool.lab_id,
      nama_alat: newTool.nama,
      merk: newTool.merk,
      tipe: newTool.tipe,
      status: AlatStatus.TERSEDIA
    });
    setNewTool({ lab_id: '', nama: '', merk: '', tipe: '' });
    setShowAddTool(false);
    loadData();
  };

  const handleDeleteTool = async (id: string) => {
    if (window.confirm("Hapus Alat ini?")) {
      try {
        await db.deleteTool(id);
        loadData();
      } catch (e: any) {
        alert("Gagal menghapus alat: " + e.message);
      }
    }
  };

  const getStatusColor = (status: AlatStatus) => {
    switch(status) {
      case AlatStatus.TERSEDIA: return 'bg-emerald-100 text-emerald-700';
      case AlatStatus.DIPINJAM: return 'bg-amber-100 text-amber-700';
      case AlatStatus.RUSAK: return 'bg-rose-100 text-rose-700';
      case AlatStatus.MAINTENANCE: return 'bg-slate-100 text-slate-700';
      default: return 'bg-gray-100';
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h2 className="text-2xl font-bold text-slate-800 flex items-center">
          <span className="bg-medical-100 text-medical-600 p-2 rounded-lg mr-3"><Wrench size={24} /></span>
          Manajemen Peralatan
        </h2>
        <div className="flex bg-slate-100 p-1 rounded-lg">
          <button 
            onClick={() => setActiveTab('list')}
            className={`px-4 py-2 rounded-md text-sm font-medium transition-all ${activeTab === 'list' ? 'bg-white shadow text-medical-600' : 'text-slate-500'}`}
          >
            Daftar Alat
          </button>
          <button 
             onClick={() => setActiveTab('calibration')}
             className={`px-4 py-2 rounded-md text-sm font-medium transition-all ${activeTab === 'calibration' ? 'bg-white shadow text-medical-600' : 'text-slate-500'}`}
          >
            Kalibrasi
          </button>
        </div>
      </div>

      {activeTab === 'list' && (
        <div className="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden">
          <div className="p-4 border-b border-slate-100 flex gap-4 justify-between items-center">
             <div className="relative flex-1 max-w-md">
               <Search className="absolute left-3 top-2.5 text-slate-400" size={18} />
               <input type="text" placeholder="Cari alat..." className="w-full pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-medical-500 outline-none" />
             </div>
             <button 
              onClick={() => setShowAddTool(!showAddTool)}
              className="bg-medical-600 text-white px-4 py-2 rounded-lg font-medium text-sm hover:bg-medical-700 transition-colors flex items-center"
             >
               <Plus size={18} className="mr-1" /> Tambah Alat
             </button>
          </div>

          {/* ADD TOOL FORM */}
          {showAddTool && (
            <div className="p-4 bg-slate-50 border-b border-slate-200">
              <form onSubmit={handleAddTool} className="grid grid-cols-1 md:grid-cols-5 gap-3 items-end">
                <div>
                   <label className="text-xs font-bold text-slate-500">Laboratorium</label>
                   <select required className="w-full text-sm p-2 border rounded" value={newTool.lab_id} onChange={e => setNewTool({...newTool, lab_id: e.target.value})}>
                     <option value="">-- Pilih Lab --</option>
                     {labs.map(l => <option key={l.id} value={l.id}>{l.nama_lab}</option>)}
                   </select>
                </div>
                <div>
                   <label className="text-xs font-bold text-slate-500">Nama Alat</label>
                   <input required className="w-full text-sm p-2 border rounded" placeholder="Ex: Mikroskop" value={newTool.nama} onChange={e => setNewTool({...newTool, nama: e.target.value})} />
                </div>
                <div>
                   <label className="text-xs font-bold text-slate-500">Merk</label>
                   <input className="w-full text-sm p-2 border rounded" placeholder="Ex: Olympus" value={newTool.merk} onChange={e => setNewTool({...newTool, merk: e.target.value})} />
                </div>
                <div>
                   <label className="text-xs font-bold text-slate-500">Tipe/Model</label>
                   <input className="w-full text-sm p-2 border rounded" placeholder="Ex: CX-21" value={newTool.tipe} onChange={e => setNewTool({...newTool, tipe: e.target.value})} />
                </div>
                <div className="flex gap-2">
                   <button type="button" onClick={() => setShowAddTool(false)} className="bg-slate-200 text-slate-600 px-3 py-2 rounded text-sm">Batal</button>
                   <button type="submit" className="bg-medical-600 text-white px-3 py-2 rounded text-sm flex-1">Simpan</button>
                </div>
              </form>
            </div>
          )}

          <table className="w-full text-left text-sm">
            <thead className="bg-slate-50 text-slate-600">
              <tr>
                <th className="p-4">Nama Alat</th>
                <th className="p-4">Lokasi</th>
                <th className="p-4">Merk / Tipe</th>
                <th className="p-4">Status</th>
                <th className="p-4">Aksi</th>
              </tr>
            </thead>
            <tbody>
              {tools.length === 0 && <tr><td colSpan={5} className="p-8 text-center text-slate-400">Belum ada data alat.</td></tr>}
              {tools.map(tool => (
                <tr key={tool.id} className="border-b border-slate-50 hover:bg-slate-50">
                  <td className="p-4 font-medium text-slate-800">{tool.nama_alat}</td>
                  <td className="p-4 text-slate-500">{labs.find(l => l.id === tool.lab_id)?.nama_lab || '-'}</td>
                  <td className="p-4 text-slate-500">{tool.merk} - {tool.tipe}</td>
                  <td className="p-4">
                    <span className={`px-2 py-1 rounded-full text-xs font-semibold ${getStatusColor(tool.status)}`}>
                      {tool.status}
                    </span>
                  </td>
                  <td className="p-4 flex gap-3">
                    {tool.status === AlatStatus.TERSEDIA && (
                      <button 
                        className="text-amber-600 hover:underline"
                        onClick={() => {
                          const peminjam = prompt("Nama Peminjam:");
                          if (peminjam) {
                            db.borrowTool(tool.id, peminjam).then(() => {
                              alert('Berhasil dipinjam');
                              loadData();
                            });
                          }
                        }}
                      >
                        Pinjam
                      </button>
                    )}
                    <button onClick={() => handleDeleteTool(tool.id)} className="text-slate-400 hover:text-red-500">
                      <Trash2 size={16} />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {activeTab === 'calibration' && (
        <div className="grid grid-cols-1 gap-6">
          <div className="bg-blue-50 border border-blue-200 p-4 rounded-lg flex items-start gap-3">
             <FileCheck className="text-blue-600 mt-1" size={20} />
             <div>
               <h4 className="font-bold text-blue-800 text-sm">Integrasi Supabase Storage</h4>
               <p className="text-xs text-blue-700 mt-1">
                 File sertifikat yang diupload akan disimpan di Bucket `certificates`. URL publik akan disimpan di database.
                 Fitur upload di bawah ini adalah simulasi UI.
               </p>
             </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border border-slate-200">
            <div className="p-6 border-b border-slate-100 flex justify-between items-center">
              <h3 className="font-bold text-slate-700">Riwayat Kalibrasi</h3>
            </div>
            <table className="w-full text-left text-sm">
            <thead className="bg-slate-50 text-slate-600">
              <tr>
                <th className="p-4">ID Alat</th>
                <th className="p-4">Tgl Kalibrasi</th>
                <th className="p-4">Berlaku Sampai</th>
                <th className="p-4">Sertifikat</th>
              </tr>
            </thead>
            <tbody>
              {calibrations.map(cal => (
                <tr key={cal.id} className="border-b border-slate-50">
                  <td className="p-4 font-mono text-slate-500">{tools.find(t => t.id === cal.alat_id)?.nama_alat || cal.alat_id}</td>
                  <td className="p-4">{new Date(cal.tgl_kalibrasi).toLocaleDateString()}</td>
                  <td className="p-4">
                    <span className={new Date(cal.tgl_expired_kalibrasi) < new Date() ? 'text-rose-600 font-bold' : ''}>
                      {new Date(cal.tgl_expired_kalibrasi).toLocaleDateString()}
                    </span>
                  </td>
                  <td className="p-4">
                    <button className="flex items-center text-xs bg-slate-100 hover:bg-slate-200 px-2 py-1 rounded text-slate-700 transition-colors">
                      <FileCheck size={14} className="mr-1" /> Lihat PDF
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
          
          <div className="p-6 bg-slate-50 border-t border-slate-200">
            <h4 className="font-bold text-sm mb-4 text-slate-700">Input Kalibrasi Baru</h4>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 items-end">
              <div>
                <label className="text-xs font-bold text-slate-500 uppercase block mb-1">Pilih Alat</label>
                <select className="w-full p-2 border rounded text-sm">
                  {tools.map(t => <option key={t.id} value={t.id}>{t.nama_alat}</option>)}
                </select>
              </div>
              <div>
                <label className="text-xs font-bold text-slate-500 uppercase block mb-1">Tgl Kalibrasi</label>
                <input type="date" className="w-full p-2 border rounded text-sm" />
              </div>
              <div>
                 <label className="text-xs font-bold text-slate-500 uppercase block mb-1">Upload Sertifikat</label>
                 <div className="relative border border-dashed border-slate-300 rounded bg-white p-1.5 flex items-center justify-center cursor-pointer hover:bg-slate-50 transition-colors">
                   <UploadCloud size={16} className="text-slate-400 mr-2" />
                   <span className="text-xs text-slate-500">Pilih File...</span>
                   <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" />
                 </div>
              </div>
              <button className="bg-medical-600 text-white p-2 rounded text-sm font-medium hover:bg-medical-700">
                Simpan Data
              </button>
            </div>
          </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default EquipmentManagement;