import React, { useEffect, useState } from 'react';
import { db } from '../services/mockSupabase';
import { StokReagenLot, KalibrasiAlat, Reagen, Alat } from '../types';
import { AlertTriangle, Clock, Activity, ShieldAlert, Download } from 'lucide-react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';

const Dashboard: React.FC = () => {
  const [loading, setLoading] = useState(true);
  const [lowStock, setLowStock] = useState<(StokReagenLot & { reagenName: string })[]>([]);
  const [expiringLots, setExpiringLots] = useState<(StokReagenLot & { reagenName: string })[]>([]);
  const [dueCalibration, setDueCalibration] = useState<(KalibrasiAlat & { alatName: string })[]>([]);
  const [stats, setStats] = useState({ totalReagens: 0, totalAlat: 0 });

  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [lots, reagents, calibrations, tools] = await Promise.all([
          db.getAllLots(),
          db.getReagents(),
          db.getAllCalibrations(),
          db.getAllTools()
        ]);

        const today = new Date();
        const thirtyDaysFromNow = new Date();
        thirtyDaysFromNow.setDate(today.getDate() + 30);

        // Process Reagents
        const lowStockItems: (StokReagenLot & { reagenName: string })[] = [];
        const expiringItems: (StokReagenLot & { reagenName: string })[] = [];

        lots.forEach(lot => {
          const reagent = reagents.find(r => r.id === lot.reagen_id);
          const name = reagent ? reagent.nama_reagen : 'Unknown';
          const expDate = new Date(lot.tgl_expired);

          // Rule: Low stock < 50 (arbitrary threshold for demo)
          if (lot.stok_sekarang < 50) {
            lowStockItems.push({ ...lot, reagenName: name });
          }

          // Rule: Expiring in 60 days or already expired
          const sixtyDays = new Date();
          sixtyDays.setDate(today.getDate() + 60);
          if (expDate <= sixtyDays) {
            expiringItems.push({ ...lot, reagenName: name });
          }
        });

        // Process Calibration
        const calibrationItems: (KalibrasiAlat & { alatName: string })[] = [];
        calibrations.forEach(cal => {
          const tool = tools.find(t => t.id === cal.alat_id);
          const name = tool ? tool.nama_alat : 'Unknown';
          const dueDate = new Date(cal.tgl_expired_kalibrasi);

          if (dueDate <= thirtyDaysFromNow) {
            calibrationItems.push({ ...cal, alatName: name });
          }
        });

        setLowStock(lowStockItems);
        setExpiringLots(expiringItems);
        setDueCalibration(calibrationItems);
        setStats({ totalReagens: reagents.length, totalAlat: tools.length });

      } catch (error) {
        console.error("Failed to fetch dashboard data", error);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const handleExport = () => {
    // Mock export functionality
    const csvContent = "data:text/csv;charset=utf-8," 
      + "Type,Name,Detail,Status\n"
      + lowStock.map(i => `Low Stock,${i.reagenName},Qty: ${i.stok_sekarang},Critical`).join("\n")
      + expiringLots.map(i => `Expiring,${i.reagenName},Date: ${i.tgl_expired},Warning`).join("\n")
      + dueCalibration.map(i => `Calibration,${i.alatName},Due: ${i.tgl_expired_kalibrasi},Action Required`).join("\n");
    
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "laporan_monitoring_lab.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  if (loading) return <div className="p-8 text-center text-slate-500">Memuat Analisis Laboratorium...</div>;

  return (
    <div className="space-y-6 animate-fade-in">
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold text-slate-800">Dashboard Monitoring</h2>
          <p className="text-slate-500 text-sm">Real-time status reagen dan peralatan laboratorium</p>
        </div>
        <button 
          onClick={handleExport}
          className="flex items-center space-x-2 bg-emerald-600 hover:bg-emerald-700 text-white px-4 py-2 rounded-lg transition-colors"
        >
          <Download size={18} />
          <span>Export Excel</span>
        </button>
      </div>

      {/* Warning Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {/* Low Stock */}
        <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200 relative overflow-hidden">
          <div className="absolute top-0 right-0 p-4 opacity-10">
            <Activity size={64} className="text-rose-600" />
          </div>
          <div className="flex items-center space-x-3 mb-4">
            <div className="p-2 bg-rose-100 text-rose-600 rounded-lg">
              <Activity size={24} />
            </div>
            <h3 className="font-semibold text-slate-700">Reagen Stok Kritis</h3>
          </div>
          <div className="text-3xl font-bold text-rose-600 mb-2">{lowStock.length}</div>
          <p className="text-sm text-slate-500">Item dengan stok di bawah threshold.</p>
          <div className="mt-4 space-y-2">
            {lowStock.slice(0, 3).map((item, idx) => (
              <div key={idx} className="flex justify-between text-sm border-b border-slate-50 pb-1">
                <span className="text-slate-700 font-medium truncate w-2/3">{item.reagenName}</span>
                <span className="text-rose-500 font-bold">{item.stok_sekarang} {item.satuan}</span>
              </div>
            ))}
            {lowStock.length > 3 && <p className="text-xs text-center text-slate-400">+{lowStock.length - 3} lainnya</p>}
          </div>
        </div>

        {/* Expiring */}
        <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200 relative overflow-hidden">
           <div className="absolute top-0 right-0 p-4 opacity-10">
            <Clock size={64} className="text-amber-600" />
          </div>
          <div className="flex items-center space-x-3 mb-4">
            <div className="p-2 bg-amber-100 text-amber-600 rounded-lg">
              <Clock size={24} />
            </div>
            <h3 className="font-semibold text-slate-700">Reagen Expired</h3>
          </div>
          <div className="text-3xl font-bold text-amber-600 mb-2">{expiringLots.length}</div>
          <p className="text-sm text-slate-500">Akan expired dalam 60 hari atau sudah lewat.</p>
          <div className="mt-4 space-y-2">
            {expiringLots.slice(0, 3).map((item, idx) => (
              <div key={idx} className="flex justify-between text-sm border-b border-slate-50 pb-1">
                <span className="text-slate-700 font-medium truncate w-2/3">{item.reagenName}</span>
                <span className="text-xs text-slate-500">{new Date(item.tgl_expired).toLocaleDateString('id-ID')}</span>
              </div>
            ))}
          </div>
        </div>

        {/* Calibration */}
        <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200 relative overflow-hidden">
           <div className="absolute top-0 right-0 p-4 opacity-10">
            <ShieldAlert size={64} className="text-medical-600" />
          </div>
          <div className="flex items-center space-x-3 mb-4">
            <div className="p-2 bg-medical-100 text-medical-600 rounded-lg">
              <ShieldAlert size={24} />
            </div>
            <h3 className="font-semibold text-slate-700">Jatuh Tempo Kalibrasi</h3>
          </div>
          <div className="text-3xl font-bold text-medical-600 mb-2">{dueCalibration.length}</div>
          <p className="text-sm text-slate-500">Alat perlu kalibrasi dalam 30 hari.</p>
           <div className="mt-4 space-y-2">
            {dueCalibration.slice(0, 3).map((item, idx) => (
              <div key={idx} className="flex justify-between text-sm border-b border-slate-50 pb-1">
                <span className="text-slate-700 font-medium truncate w-2/3">{item.alatName}</span>
                <span className="text-xs text-rose-500 font-medium">H-{(new Date(item.tgl_expired_kalibrasi).getTime() - new Date().getTime()) / (1000 * 3600 * 24) < 0 ? 'Overdue' : Math.ceil((new Date(item.tgl_expired_kalibrasi).getTime() - new Date().getTime()) / (1000 * 3600 * 24)) }</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200">
        <h3 className="font-semibold text-slate-700 mb-4">Status Aset Laboratorium</h3>
        {/* Fixed height container to prevent Recharts error */}
        <div className="w-full h-72">
          <ResponsiveContainer width="100%" height="100%">
            <BarChart
              data={[
                { name: 'Total Reagen', value: stats.totalReagens },
                { name: 'Stok Kritis', value: lowStock.length },
                { name: 'Expired', value: expiringLots.length },
                { name: 'Total Alat', value: stats.totalAlat },
                { name: 'Perlu Kalibrasi', value: dueCalibration.length },
              ]}
              layout="vertical"
              margin={{ top: 5, right: 30, left: 20, bottom: 5 }}
            >
              <CartesianGrid strokeDasharray="3 3" horizontal={false} />
              <XAxis type="number" />
              <YAxis dataKey="name" type="category" width={100} />
              <Tooltip />
              <Legend />
              <Bar dataKey="value" fill="#0284c7" barSize={30} radius={[0, 4, 4, 0]} />
            </BarChart>
          </ResponsiveContainer>
        </div>
      </div>
    </div>
  );
};

export default Dashboard;