import React, { useState, useEffect } from 'react';
import { supabase } from './services/mockSupabase';
import { Session } from '@supabase/supabase-js';
import Sidebar from './components/Sidebar';
import Dashboard from './components/Dashboard';
import ReagentManagement from './components/ReagentManagement';
import EquipmentManagement from './components/EquipmentManagement';
import LabManagement from './components/LabManagement';
import SqlDocumentation from './components/SqlDocumentation';
import Login from './components/Login';
import { Activity } from 'lucide-react';

const App: React.FC = () => {
  const [currentView, setCurrentView] = useState('dashboard');
  const [session, setSession] = useState<Session | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    // Check active session
    supabase.auth.getSession().then(({ data: { session } }) => {
      setSession(session);
      setLoading(false);
    });

    // Listen for auth changes
    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      setSession(session);
    });

    return () => subscription.unsubscribe();
  }, []);

  const renderContent = () => {
    switch (currentView) {
      case 'dashboard':
        return <Dashboard />;
      case 'reagens':
        return <ReagentManagement />;
      case 'alat':
        return <EquipmentManagement />;
      case 'labs':
        return <LabManagement />;
      case 'doc':
        return <SqlDocumentation />;
      default:
        return <Dashboard />;
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-slate-50 flex items-center justify-center">
        <div className="text-center">
          <Activity className="h-10 w-10 text-medical-600 animate-bounce mx-auto mb-4" />
          <p className="text-slate-500 font-medium">Memuat Sistem...</p>
        </div>
      </div>
    );
  }

  if (!session) {
    return <Login />;
  }

  return (
    <div className="flex min-h-screen bg-slate-50 font-sans">
      <Sidebar currentView={currentView} setCurrentView={setCurrentView} userEmail={session.user.email} />
      
      <main className="flex-1 ml-64 p-8 overflow-y-auto">
        <header className="flex justify-between items-center mb-8">
          <div>
            <h1 className="text-2xl font-bold text-slate-800">
              {currentView === 'dashboard' && 'Overview'}
              {currentView === 'reagens' && 'Manajemen Stok Reagen'}
              {currentView === 'alat' && 'Manajemen Alat & Kalibrasi'}
              {currentView === 'labs' && 'Data Laboratorium'}
              {currentView === 'doc' && 'Dokumentasi Sistem'}
            </h1>
            <p className="text-sm text-slate-500">Selamat datang, {session.user.email}</p>
          </div>
          <div className="flex items-center space-x-4">
            <span className="text-xs text-slate-400">{new Date().toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}</span>
          </div>
        </header>

        {renderContent()}
      </main>
    </div>
  );
};

export default App;